import SwiftUI
import Foundation

/// Coordinator to manage disclaimer presentation flow
/// Ensures App Store compliance by presenting disclaimer before app access
struct DisclaimerCoordinator: ViewModifier {
    @StateObject private var disclaimerService = DisclaimerService()
    @State private var showingDisclaimer = false
    
    func body(content: Content) -> some View {
        content
            .fullScreenCover(isPresented: $showingDisclaimer) {
                NavigationView {
                    DisclaimerView(hasAcceptedDisclaimer: $disclaimerService.hasAcceptedDisclaimer)
                        .interactiveDismissDisabled(true) // Prevent dismissal without acceptance
                }
            }
            .onAppear {
                checkDisclaimerRequirement()
            }
            .onChange(of: disclaimerService.hasAcceptedDisclaimer) { accepted in
                if accepted {
                    showingDisclaimer = false
                }
            }
            .environmentObject(disclaimerService)
    }
    
    private func checkDisclaimerRequirement() {
        if disclaimerService.shouldShowDisclaimer {
            // Small delay to ensure smooth presentation
            DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
                showingDisclaimer = true
            }
        }
    }
}

// Extension to make it easy to apply disclaimer coordination
extension View {
    /// Apply medical disclaimer coordination to ensure App Store compliance
    func withDisclaimerCompliance() -> some View {
        self.modifier(DisclaimerCoordinator())
    }
}

/// Wrapper view for the entire app that ensures disclaimer compliance
struct DisclaimerComplianceWrapper<Content: View>: View {
    let content: Content
    
    init(@ViewBuilder content: () -> Content) {
        self.content = content()
    }
    
    var body: some View {
        content
            .withDisclaimerCompliance()
    }
}